package gov.va.med.mhv.vitals.validator;

import java.math.BigDecimal;
import java.util.Date;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import gov.va.med.mhv.common.api.util.ResponseUtil;
import gov.va.med.mhv.vitals.dto.InrReadingDTO;
import gov.va.med.mhv.vitals.enums.InrHighEndTargetRangeEnumeration;
import gov.va.med.mhv.vitals.enums.InrLowEndTargetRangeEnumeration;
import gov.va.med.mhv.vitals.util.MessagesUtil;

@Component
public class InrValidator extends Validator{
	
	public static final String LOW_END =  "lowEndTargetRange";
	public static final String HIGH_END = "highEndTargetRange";
	public static final String NONE_VALUE = "NONE";
	public static final int COMMENTS_MAX_LENGTH = 255;
	private static final BigDecimal INRVALUE_MIN_VALUE = new BigDecimal("0.1");
	private static final BigDecimal INRVALUE_MAX_VALUE = new BigDecimal("99.9");
	private static final int INRVALUE_SCALE = 1;
	public static final int PROVIDER_MAX_LENGTH = 100;
	public static final int LOCATION_MAX_LENGTH = 100;

	
	@Autowired
	private MessagesUtil messages;
	
	public void inrReadingValidations(ResponseUtil response, InrReadingDTO inrdto) {
		fieldValidation(response, inrdto);
		if (response.getValidationErrors().size() == 0) {
			complexValidation(response, inrdto);
		}
	}
	
	private void complexValidation(ResponseUtil response, InrReadingDTO inrdto) {
		if(inrdto.getLowendTargetRange().equals(NONE_VALUE) && !NONE_VALUE.equals(inrdto.getHighendTragetRange())){
			response.getValidationErrors().put("sei.inr.detail.error.invalidValue.lowHighEndtargetRange",
					messages.getInvalidInrRange());
		}else if(!NONE_VALUE.equals(inrdto.getLowendTargetRange()) && inrdto.getHighendTragetRange().equals(NONE_VALUE)){
			response.getValidationErrors().put("sei.inr.detail.error.invalidValue.lowHighEndtargetRange",
					messages.getInvalidInrRange());
		}
	}



	private void fieldValidation(ResponseUtil response, InrReadingDTO inrdto) {
		validateComments(response,inrdto);
		validateLowEndTargetRange(response,inrdto);
		validateInrValue(response,inrdto);
		validateDateEntered(response,inrdto);
		validateProvider(response,inrdto);
		validateLocation(response,inrdto);
		validateHighEndTargetRange(response,inrdto);
	 }

	private void validateHighEndTargetRange(ResponseUtil response,InrReadingDTO inrdto) {
		String value = inrdto.getHighendTragetRange();
		
		if ( value == null ) {
			Validations.validateRequired("highEndTargetRange", value, response.getValidationErrors());
			return;
		}	
		Validations.validateEnumeration("highEndTargetRange", value, InrHighEndTargetRangeEnumeration.values(), response.getValidationErrors());
		
	}

	private void validateLocation(ResponseUtil response, InrReadingDTO inrdto) {
		String value = inrdto.getLocation();
		if ( value == null ) {
			return;
		}						
		Validations.validateMaxLength("Location",value, LOCATION_MAX_LENGTH, response.getValidationErrors());

	}

	private void validateProvider(ResponseUtil response, InrReadingDTO inrdto) {
		String value = inrdto.getProvider();
		if ( value == null ) {
			return;
		}						
		Validations.validateMaxLength("Provider",value, PROVIDER_MAX_LENGTH, response.getValidationErrors());
	}

	private void validateDateEntered(ResponseUtil response, InrReadingDTO inrdto) {
		Date value = inrdto.getReading();
		if( value == null ) {
			Validations.validateRequired("dateEntered", value, response.getValidationErrors());
			return;
		}		
	}

	private void validateInrValue(ResponseUtil response, InrReadingDTO inrdto) {
		
		BigDecimal value = inrdto.getInr();
		if( value == null ) {
			Validations.validateRequired("INR value", value, response.getValidationErrors());
			return;
		}		
		Validations.validateMinValue("INR value minumum", value, INRVALUE_MIN_VALUE,"The INR value ", response.getValidationErrors());
		Validations.validateMaxValue("INR value maximum",value, INRVALUE_MAX_VALUE,"The INR value ", response.getValidationErrors());
		Validations.validateScale("INR value scale",value, INRVALUE_SCALE, "The INR value ", response.getValidationErrors());
	}

	private void validateLowEndTargetRange(ResponseUtil response,InrReadingDTO inrdto) {
		
		String value = inrdto.getLowendTargetRange();
		
		if ( value == null ) {
			Validations.validateRequired("lowEndTargetRange", value, response.getValidationErrors());
			return;
		}	
		Validations.validateEnumeration("lowEndTargetRange", value, InrLowEndTargetRangeEnumeration.values(), response.getValidationErrors());
	}

	private void validateComments(ResponseUtil response, InrReadingDTO inrdto) {
		String value = inrdto.getComments();
		if ( value == null ) {
			return;
		}						
		Validations.validateMaxLength("comments",value, COMMENTS_MAX_LENGTH, response.getValidationErrors());
	}

}
